import os, subprocess, tempfile

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# Paths
PROJECT   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Baseball-Animation-1.aep"
CSV_PATH  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_Baseball-Homerun-1.jsx"

# AE comp
COMP = "HomeRun-192"

# Naming knobs you can change per run
RUN_TAG   = "H"      # e.g., "R", "TD", "FG", "ALT"
ANIM_TAIL = "_1_3"   # will be appended (leading "_" auto-added if missing)

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Homerun_Animation_1_PNG-192"
PATH_TEMPLATE = "{league}"      # <— league only (no team_id segment)
OM_TEMPLATE   = "PNG Sequence"  # must exist in AE Output Module templates

RUN_MODE = "league"             # or "subset"
LEAGUE   = "MLB"
SUBSET   = ["PHI","NYM","LAD"]  # only used if RUN_MODE == "subset"

def _write_bridge_jsx(env):
    def esc(s): return s.replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(str(v))+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines)+"\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def run(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa = [
        "osascript",
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to activate',
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to DoScriptFile POSIX file "{bridge}"',
    ]
    subprocess.run(osa, check=True)
    print("\nDone. Output ->", OUTDIR)

def make_env(league=None, teams=None):
    env = {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,

        # Layers
        "AE_TEAMNAME_LAYER": "TeamName",
        "AE_TEAMLOGO_LAYER": "TeamLogo",
        "AE_STREAKS_LAYER": "Streaks",
        "AE_LARGEBASEBALL_LAYER": "LargeBaseball",

        # Baseball sequence root (without -N.png)
        "AE_BASEBALL_PATH": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/BaseBall/HomeRunAnimation_1/large baseball/Large-BB",

        # Logos
        "AE_LOGO_DIR": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large/MLB",
        "AE_LOGO_EXTS": "png,jpg,jpeg,svg,ai,psd",

        # Output / templates
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,  # supports {league} and {abbr} (no team_id)
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        # Naming controls (drive the animName)
        "AE_RUN_TAG": RUN_TAG,
        "AE_ANIM_TAIL": ANIM_TAIL,

        # General
        "AE_NO_RENDER": "0",
        "AE_QUIT": "1",
        "AE_PURGE_BEFORE_RENDER": "1",
        "AE_LEAGUE": league or "",
    }
    if teams:  env["AE_TEAMS"]  = ",".join(teams)
    return env

if __name__ == "__main__":
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"
    env = make_env(league=LEAGUE) if RUN_MODE=="league" else make_env(teams=SUBSET)
    run(env)
